# -*- coding: utf-8 -*-

from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from time import sleep
from datetime import datetime
import os
import sys
import smtplib
import configparser as cp
import requests as req
import logging as log
from logging.handlers import TimedRotatingFileHandler
from daemonize import Daemonize

my_path = os.path.abspath(os.path.dirname(__file__))

def send_mail(smtp, port, from_email, password, to_email, subject, message):
    try:
        msg = MIMEMultipart()
        msg['from'] = from_email
        msg['to'] = to_email
        msg['subject'] = subject
        msg.attach(MIMEText(message, 'plain'))

        server = smtplib.SMTP('%s: %s' % (smtp, port))
        server.starttls()
        server.login(msg['from'], password)
        server.sendmail(msg['from'], msg['to'], msg.as_string())

        server.quit()

        log.info('E-mail enviado com sucesso para %s:' % (msg['to']))
    except Exception as e:
        log.error('Falha ao enviar o e-mail: %s' % e)

def recovery(commands):
    commands_list = commands.split(';')
    for cmd in commands_list:
        #log.info('# %s' % cmd)
        os.system(cmd)
        sleep(3)

def create_msg(msg, request_obj, app_name):
    current_datetime = datetime.today().strftime('%d/%m/%Y %H:%M:%S')

    message = msg
    message += ('\n\n=========== INFO ===========\n\n')
    message += ('- App: %s\n' % app_name)
    message += ('- URL: %s\n' % request_obj['url'])
    message += ('- Timeout: %s\n' % request_obj['timeout'])
    message += ('- Falhas consecutivas: %s' % request_obj['expected_falls'])
    message += ('\n\n%s' % current_datetime)

    return message

def request(url, timeout):
    try:
        resp = req.get(url, timeout = float(timeout))
        return resp
    except Exception as e:
        log.error(e)

def log_config():
    logFormatter = log.Formatter('%(asctime)s %(levelname)-8s %(message)s')
    rootLogger = log.getLogger()
    rootLogger.setLevel(log.DEBUG)

    path_logs = os.path.join(my_path, 'logs/healthcheck.log')
    timedRotatingFileHandler = TimedRotatingFileHandler(path_logs, when = 'D', interval = 1)
    timedRotatingFileHandler.setFormatter(logFormatter)
    rootLogger.addHandler(timedRotatingFileHandler)

    consoleHandler = log.StreamHandler()
    consoleHandler.setFormatter(logFormatter)
    rootLogger.addHandler(consoleHandler)

def main():
    log_config()

    while 1:
        log.info('=========== * Iniciando o healthcheck * ===========')

        conf = cp.ConfigParser()
        path_conf = os.path.join(my_path, 'conf.cfg')
        conf.read(r'%s' % path_conf)

        app_obj = {}
        app_obj['name'] = conf['app']['name']

        email_obj = {}
        email_obj['smtp'] = conf['email']['smtp']
        email_obj['port'] = conf['email']['port']
        email_obj['from_email'] = conf['email']['from_email']
        email_obj['password'] = conf['email']['password']
        email_obj['to_email'] = conf['email']['to_email']
        email_obj['subject'] = conf['email']['subject']
        email_obj['message'] = conf['email']['message']

        request_obj = {}
        request_obj['url'] = conf['request']['url']
        request_obj['timeout'] = conf['request']['timeout']
        request_obj['expected_falls'] = conf['request']['expected_falls']
        request_obj['seconds_initial_wait'] = conf['request']['seconds_initial_wait']
        request_obj['seconds_interval'] = conf['request']['seconds_interval']
        request_obj['response_value'] = conf['request']['response_value']

        recovery_obj = {}
        recovery_obj['commands'] = conf['recovery']['commands']
        recovery_obj['reload_app'] = conf['recovery']['reload_app']

        log.info('- App: %s' % app_obj['name'])
        log.info('- URL: %s' % request_obj['url'])
        log.info('- Timeout: %s' % request_obj['timeout'])
        log.info('- Falhas permitidas: %s' % request_obj['expected_falls'])

        log.info('Esperando %s segundos ate iniciar o processo de healthcheck...' % request_obj['seconds_initial_wait'])
        sleep(int(request_obj['seconds_initial_wait']))

        falls = 0

        while falls < int(request_obj['expected_falls']):
            resp = request(request_obj['url'], request_obj['timeout'])
            if resp is not None and resp.status_code == 200:
                try:
                    data = ''
                    try:
                        data = resp.json()
                    except ValueError:
                        data = resp.text
                    if str(data) == str(request_obj['response_value']):
                        log.info('Sucesso no healthcheck. [Status: %s] [Dados: %s]' % (resp, data))
                    else:
                        log.warning('O healthcheck retornou sucesso porem os dados retornados sao diferentes do esperado. [Status: %s] [Dados retornados: %s] [Dados esperados: %s]' % (resp, data, request_obj['response_value']))
                    falls = 0
                except Exception as e:
                    log.warning('O healthcheck retornou sucesso porem os dados nao puderam ser processados. [Status: %s] [Erro: %s]' % (resp, e))
            else:
                falls += 1
                erro = resp
                if resp is None:
                    erro = 'Timeout exceeded'
                log.warning('O healthcheck falhou %d de %s vezes permitidas. [Erro: %s]' % (falls, request_obj['expected_falls'], erro))
            if falls < int(request_obj['expected_falls']):
                log.info('Esperando %s segundos ate a proxima requisicao...' % request_obj['seconds_interval'])
                sleep(int(request_obj['seconds_interval']))

        log.critical(('O numero maximo de falhas foi atingido, iniciando processo de recuperacao...').upper())

        message = create_msg(email_obj['message'], request_obj, app_obj['name'])
        send_mail(email_obj['smtp'], email_obj['port'], email_obj['from_email'], email_obj['password'], email_obj['to_email'], email_obj['subject'], message)

        log.info('=========== * Healthcheck finalizado * ===========')

        recovery(recovery_obj['commands'])

        if str(recovery_obj['reload_app']) != '1':
            break

    sys.exit(0)

if __name__== '__main__':
    daemon = Daemonize(app='healthcheck', pid='/tmp/healthcheck.pid', action=main)
    daemon.start()

